/************************************************************************
 * NAME:	svd12.c
 *
 * DESCR:	Support for version 1.2 of the SVD format.  You should
 *		also look at svd.[ch] and svd[0-9][0-9].[ch] which have
 *		code for the different versions of the SVD format.
 *		These were split out from this file to eliminate
 *		confusion and inadvertant changes to the old formats.
 ************************************************************************/
#include <sys/types.h>
#include <stdio.h>

#include "standard.h"
#include "floppy.h"
#include "error.h"
#include "svd.h"

/************************************************************************
 * NAME:	svd_read_12header()
 *
 * DESCR:	Read the v1.2 header information.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
svd_read_12header(int fd, int *sectors, int *tracks)
{
    char    linebuffer[80];
    char   *ptr;
    int	    count;

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d",sectors) != 1) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    count = svd_read_line(fd,linebuffer,sizeof(linebuffer));
    if (count == 0) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }
    if (sscanf(linebuffer,"%d",tracks) != 1) {
	return(E_FATAL|E_SVD_BAD_FORMAT);
    }

    return(E_NONE);
}

/************************************************************************
 * NAME:	svd_read_12()
 *
 * DESCR:	Read in a version 1.2 of an SVD file.  Picks up right
 *		after a version number was found.
 *
 * ARGS:	
 *
 * RETURNS:	
 *
 * NOTES:	
 ************************************************************************/
int
svd_read_12(int fd,struct floppy *floppy)
{
    int sectors, tracks;
    int side, sector, track;
    int	r;
    unsigned char blockbuf[256];
    int	 indexptr;

    if ( (r=svd_read_12header(fd,&sectors,&tracks)) != E_NONE) {
	return(r);
    }

    /* ok, now we have (sectors+1) * (tracks) of 256-byte blocks coming up	*/

    floppy->sectors = sectors;
    floppy->tracks = tracks;
    floppy->sides = 1;		/* note that this is really ignored in 1.2	*/
    floppy->encoding = WD_FM;
    floppy->write_protect = 0x00;

    /* allocate the track storage	*/

    floppy->side[0] = (struct track *)malloc(floppy->tracks*sizeof(struct track));

    if (floppy->sides == 2) {	/* really ignored in 1.2...just for the future	*/
	floppy->side[1] = (struct track *)malloc(floppy->tracks*sizeof(struct track));
    }

    /* allocate the sectors in each track	*/
    
    for (side = 0; side < floppy->sides; side++) {
	for (track = 0; track < floppy->tracks; track++) {
	    floppy->side[side][track].sectors = floppy->sectors;
	    floppy->side[side][track].sector = (struct sector *)
	                         malloc(floppy->side[side][track].sectors*sizeof(struct sector));
	}
    }

    /* now cruise through the blocks reading in data	*/

#define THIS_SECTOR	floppy->side[side][track].sector[sector]

    for (track = 0; track < floppy->tracks; track++) {
	for (side = 0; side < floppy->sides; side++) {

	    /* first, dissect the index block for this track	*/

	    if (read(fd,blockbuf,256) != 256) {
		return(E_FATAL|E_SVD_BAD_FORMAT);
	    }

	    indexptr = 0;

	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {
		THIS_SECTOR.id = blockbuf[indexptr++];
		THIS_SECTOR.side = blockbuf[indexptr++];
		THIS_SECTOR.sector = blockbuf[indexptr++];
		THIS_SECTOR.sizecode = blockbuf[indexptr++];
		THIS_SECTOR.headCRC = blockbuf[indexptr++];
		THIS_SECTOR.headCRC += (int)(blockbuf[indexptr++]) * 256;
		THIS_SECTOR.mark = blockbuf[indexptr++];
		THIS_SECTOR.dataCRC = blockbuf[indexptr++];
		THIS_SECTOR.dataCRC += (int)(blockbuf[indexptr++]) * 256;
		THIS_SECTOR.encoding = WD_FM;
		THIS_SECTOR.size = 256;
	    }

	    /* now read in the data for each of the blocks in this track	*/

	    indexptr = HEADER_FIELDS_TO_CRC12;	/* start past the first header fields	*/
	                        		/* but before the data CRC		*/

	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {
		int i, j;

		if (read(fd,blockbuf,256) != 256) {
		    return(E_FATAL|E_SVD_BAD_FORMAT);
		}

		j = 0;
		for (i=indexptr; i < 256; i++) {
		    THIS_SECTOR.data[j++] = blockbuf[i];
		}
		for (i = 0; i < indexptr; i++) {
		    THIS_SECTOR.data[j++] = blockbuf[i];
		}
		indexptr += HEADER_FIELDS_12;	/* jump to next header fields	*/
	    }
	}
    }
    return(E_NONE);
}

/************************************************************************
 * NAME:	svd_dump_12()
 *
 * DESCR:	Dumps out the floppy information in SVD format.
 *
 * ARGS:	fd is the file descriptor to dump on
 *		do_slide is a boolean indicating whether slide is used
 *
 * RETURNS:	0 if things OK, an error number otherwise
 *
 * NOTES:	Currently assumes single density only.
 *
 *		BIG NOTE - some things aren't currently in the file:
 *			write-protect, sides, sector/floppy density (v1.2)
 ************************************************************************/
int
svd_dump_12(struct floppy *floppy, int fd)
{
    /* this will dump out the existing floppy data in SVD format	*/

    /* for each track, dump out sectors+1 with the proper rotation	*/

    int			side;
    int			track;
    int			sector;
    int			retvalue = 0;
    int			indexptr;
    unsigned char	block[256];
    unsigned char	buffer[100];

    sprintf(buffer,"%s\n","1.2");	    write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->sectors); write(fd,buffer,strlen(buffer));
    sprintf(buffer,"%d\n",floppy->tracks);  write(fd,buffer,strlen(buffer));

    for (track = 0; track < floppy->tracks; track++) {

	for (side = 0; side < floppy->sides; side++) {

#define THIS_SECTOR	floppy->side[side][track].sector[sector]

/***	    if (THIS_SECTOR.density != 0 ) {
		error(E_SVD_UNSUP,E_SVD_UNSUPTEXT,"double density sector");
		return(E_SVD_UNSUP);
	    }
***/
	    { 
		int	i;

		for (i=0; i < 256; i++) {	/* ensures that an output file	*/
		    block[i] = 0;		/*   is always the same		*/
		}
	    }

	    indexptr = 0;

	    /* first compose the indexblock for the track	*/

	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {
		block[indexptr++] = THIS_SECTOR.id;
		block[indexptr++] = THIS_SECTOR.side;
		block[indexptr++] = THIS_SECTOR.sector;
		block[indexptr++] = THIS_SECTOR.sizecode;
		block[indexptr++] = THIS_SECTOR.headCRC & 0xff;
		block[indexptr++] = THIS_SECTOR.headCRC / 256;
		block[indexptr++] = THIS_SECTOR.mark;
		block[indexptr++] = THIS_SECTOR.dataCRC & 0xff;
		block[indexptr++] = THIS_SECTOR.dataCRC / 256;
	    }

	    write(fd,block,256);

	    /* now compose and write out sectors */

	    indexptr = HEADER_FIELDS_TO_CRC12;	/* start past the first header fields	*/
	                        		/* but before the data CRC		*/

	    for (sector = 0; sector < floppy->side[side][track].sectors; sector++) {
		int	i, j;

		j = 0;
		for (i=indexptr; i < 256; i++) {
		    block[i] = THIS_SECTOR.data[j++];
		}
		for (i = 0; i < indexptr; i++) {
		    block[i] = THIS_SECTOR.data[j++];
		}

		write(fd,block,256);
		indexptr += HEADER_FIELDS_12;		/* jump to next header fields	*/
	    }
	}
    }
    return(0);
}
